#if !defined( _AMBIENT_LIGHTING_GLOBALS_FXH )
#define _AMBIENT_LIGHTING_GLOBALS_FXH

//---------------------------------------------------------------------------------
//
// HLSL header file contains macros and definitions for specialisable ambient lighting.
//


//---------------------------------------------------------------------------------
//
// 3DSMAX version:
//
// Fixed 50%/10% grey simple hemispherical ambient lighting.
//

#if defined( _3DSMAX_ )

	#define DECLARE_AMBIENT_LIGHTING_PARAMS

	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )
	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ )

	#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS																									\
		float3 CHLworldNormal = normal;																														\
		float hemiFactor = ( CHLworldNormal.z + 1.0f ) * 0.5f;																		\
		_output.colour *= lerp( float4( 0.1, 0.1, 0.1, 1.0f ),														\
														float4( 0.5, 0.5, 0.5, 1.0f ), hemiFactor );

	#define	DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION	DO_VS_AMBIENT_LIGHTING_CALCULATIONS

	#define DO_PS_AMBIENT_LIGHTING_CALCULATIONS( __accumulator__ )


#else

	// SSAO
	#if !defined(_PS3_) && !defined(_NO_SSAO_)
		#if defined(_SSAO_READY_)	// temp define as not all shader files are ready
			#if defined( _AMBIENT_SIMPLE_HEMI_SSAO_ )	// only use SSAO with this ambient type
				#define _SSAO_
			#endif
		#endif
	#endif


#if defined(_PS3_)
	#define SSAO_CLAMP_OP		ClampToEdge
	#define	SSAO_LOD_BIAS_WORD	LODBias
#else
	#define SSAO_CLAMP_OP		Clamp
	#define	SSAO_LOD_BIAS_WORD	MipMapLODBias
#endif

#if defined(SET_FX_SAMPLER_STATES)
	#define	SSAOTEX_SAMPLER_STATES	\
		FX_SAMPLERSTATE_LINEAR_TEXTURE\
		MinFilter = Point;\
		MagFilter = Point;\
		MipFilter = Point;\
		AddressU  = SSAO_CLAMP_OP;\
		AddressV  = SSAO_CLAMP_OP;\
		SSAO_LOD_BIAS_WORD = 0;\
		SET_NO_ANISOTROPY
#else
	#define	SSAOTEX_SAMPLER_STATES
#endif

#if defined(_SSAO_)
	#define DECLARE_SSAO_PARAMS	\
		float4 SSAORenderWidthHeight						\
		<													\
			bool appEdit = false;							\
		>;													\
				\
		texture SSAOTexture : SSAOTexture	\
		<	\
			bool appEdit = false;	\
		>;		\
		sampler2D SSAOMap : SAMPLER		\
		<	\
			SET_LINEAR_TEXTURE		\
			bool appEdit = false;	\
			string SamplerTexture="SSAOTexture";	\
			string MinFilter = "linear";	\
			string MagFilter = "linear";\
			string MipFilter = "linear";\
			string AddressU  = "Clamp";\
			string AddressV  = "Clamp";\
			int MipMapLODBias = 0;\
		> \
		= sampler_state\
		{\
			Texture = < SSAOTexture >;\
			SSAOTEX_SAMPLER_STATES\
		};
#else
	#define DECLARE_SSAO_PARAMS
#endif


//---------------------------------------------------------------------------------
//
// Engine versions:
//
// Exact implementation depends on the ambient lighting define passed to the compiler.
//
// Currently defined: 
//		_AMBIENT_UNIFORM_										- uniform ambient lighting
//		_AMBIENT_SIMPLE_HEMI_								- simple hemispherical lighting
//		_AMBIENT_SIMPLE_HEMI_SSAO_							- simple hemispherical lighting with screen space ambient occlusion
//		_AMBIENT_PRECALCULATED_GLOBAL_			- ambient lighting is prebaked into color0
//		_AMBIENT_OMNIDIRECTIONAL_OCCLUSION_ - ambient occlusion in color0 with incident ambient cube
//
// Anything else gets no ambient lighting.
//

#if defined( _AMBIENT_UNIFORM_ ) // AMBIENT TYPE

//
// Uniform ambient light
//
// Global illumination is represented with just one overall colour per primitive.
//

#define DECLARE_AMBIENT_LIGHTING_PARAMS	\
float4 ambientColour : Ambient					\
<																				\
	bool appEdit = false;									\
>;

#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )
#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ )

#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS	\
	LPFLOAT ShadowLightLevel = Rv( 1.0 );			\
	_output.colour *= ambientColour;

#define	DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION	\
	LPFLOAT ShadowLightLevel = Rv( 1.0 );									\
	LPFLOAT ShadowTransition = Rv( 0.0 );									\
	_output.colour *= ambientColour;


#define DO_PS_AMBIENT_LIGHTING_CALCULATIONS( __accumulator__ )


#elif defined( _AMBIENT_SIMPLE_HEMI_ ) || defined( _AMBIENT_SIMPLE_HEMI_SSAO_ ) // AMBIENT TYPE

	//
	// Hemispherical ambient light
	//
	// Global illumination is represented with two colours, a skylight and a groundlight.
	// The y component of the vertex normal is used to interpolate between these.
	//

	#if !defined(_LIGHTING_DYNAMIC_)

		#define DECLARE_AMBIENT_LIGHTING_PARAMS	\
		float4 upColour : UpColour				\
		<										\
			bool appEdit = false;				\
		>;										\
												\
		float4 downColour : DownColour			\
		<										\
			bool appEdit = false;				\
		>;	\
		DECLARE_SSAO_PARAMS

	#else
	
			#define DECLARE_AMBIENT_LIGHTING_PARAMS		\
														\
			float2 cLightingTransitionValues : LightingTransitionValues	\
			<													\
				bool appEdit = false;							\
			>;													\
																\
			float4 upColour : UpColour							\
			<													\
				bool appEdit = false;							\
			>;													\
																\
			float4 downColour : DownColour						\
			<													\
				bool appEdit = false;							\
			>;													\
			DECLARE_SSAO_PARAMS
		
	#endif

	#if defined(_SSAO_)
		#ifdef _XBOX
			#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )	\
											float2 screen_pos : POSITION1;
		#else
			#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )
		#endif
	#else
		#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )
	#endif
	
	#if defined(_SSAO_)
		#ifdef _XBOX
			#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ ) \
									float2 screen_pos : POSITION1;
		#else
			#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ ) \
									float2 screen_pos : VPOS;
		#endif
	#else
		#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ )
	#endif		

	#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS \
		LPFLOAT3 CHLworldNormal = normal;															\
		LPFLOAT hemiFactor = ( CHLworldNormal.y + Rv(1.0) ) * Rv(0.5);			\
		_output.colour *= lerp( downColour, upColour, hemiFactor );	\
		LPFLOAT ShadowLightLevel = Rv( 1.0 );	\
		LPFLOAT ShadowTransition = Rv( 0.0 );

	#if defined(_LIGHTING_DYNAMIC_)

		#if defined(_SSAO_) && defined(_XBOX)
			#define DO_SSAO_VS_CALC		_output.screen_pos = ((_output.position.xy/_output.position.w)+1.0f)*0.5f;
		#else
			#define DO_SSAO_VS_CALC
		#endif

		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION \
			DO_SSAO_VS_CALC	\
			LPFLOAT3 CHLworldNormal = normal;															\
			LPFLOAT hemiFactor = ( CHLworldNormal.y + Rv(1.0) ) * Rv(0.5);			\
			_output.colour *= lerp( downColour, upColour, hemiFactor );	\
			_output.eye.w = 1.0f;/*(worldPos.x + worldPos.y) - (2.0*floor((worldPos.x + worldPos.y)*0.5)) > 1.0 ? 1.0 : 0.0;*/

	#elif defined(_LIGHTING_PRECALCULATED_)

		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION	DO_VS_AMBIENT_LIGHTING_CALCULATIONS

	#else

		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION	DO_VS_AMBIENT_LIGHTING_CALCULATIONS

	#endif

	#if defined(_SSAO_)
		#define DO_PS_AMBIENT_LIGHTING_CALCULATIONS( __accumulator__ )	\
			__accumulator__ *= tex2D( SSAOMap, float2(_input.screen_pos.x/SSAORenderWidthHeight.x,_input.screen_pos.y/SSAORenderWidthHeight.y) ).r;
	#else
		#define DO_PS_AMBIENT_LIGHTING_CALCULATIONS( __accumulator__ )
	#endif

#elif defined( _AMBIENT_PRECALCULATED_GLOBAL_ ) // AMBIENT TYPE

	//
	//
	//

	#if defined(_LIGHTING_DYNAMIC_)

		#define DECLARE_AMBIENT_LIGHTING_PARAMS	\
			float2 cLightingTransitionValues : LightingTransitionValues	\
			<																				\
				bool appEdit = false;									\
			>;

	#else

		#define DECLARE_AMBIENT_LIGHTING_PARAMS

	#endif

	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )
	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ )

	#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS \
		_output.colour.xyz = _input.colour.xyz; \
		LPFLOAT ShadowLightLevel = _input.colour.a >= 0.5 ? Rv( 1.0 ) : Rv( 0.0 ); \
		_output.colour.a   = (_input.colour.a * 2.0f) - ShadowLightLevel; \

	#if defined(_LIGHTING_DYNAMIC_)
		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION \
			_output.colour.xyz   = _input.colour.xyz; \
			_output.eye.w = _input.colour.a >= 0.5 ? 1.0 : 0.0;	\
			_output.colour.a   = (_input.colour.a * 2.0f) - _output.eye.w;
	#else
		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION	DO_VS_AMBIENT_LIGHTING_CALCULATIONS
	#endif

	#define DO_PS_AMBIENT_LIGHTING_CALCULATIONS( __accumulator__ )

#elif defined( _AMBIENT_OMNIDIRECTIONAL_OCCLUSION_ ) // AMBIENT TYPE


	//
	//
	//

	#if defined(_LIGHTING_DYNAMIC_)

		#define DECLARE_AMBIENT_LIGHTING_PARAMS	\
			float2 cLightingTransitionValues : LightingTransitionValues	\
			<																				\
				bool appEdit = false;									\
			>;																			\
																							\
			float4 cAmbientCubeColour[6] : AmbientCubeColour \
			<																				\
				int coherenceType = ECT_DYNAMIC;	\
				bool appEdit = false;									\
			>;

	#else

		#define DECLARE_AMBIENT_LIGHTING_PARAMS	\
			float4 cAmbientCubeColour[6] : AmbientCubeColour \
			<																				\
				int coherenceType = ECT_DYNAMIC;	\
				bool appEdit = false;									\
			>;

	#endif

	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )
	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ )

	#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS \
		/*_output.colour = _input.colour;*/ \
		_output.colour = float4( 1.0f, 1.0f, 1.0f, 1.0f ); \
		LPFLOAT3 normSquared = normal * normal; \
		int3 normIsNegative = ( normal < 0.0f ); \
		_output.colour.xyz *=	normSquared.x * cAmbientCubeColour[ normIsNegative.x ].xyz + \
													normSquared.y * cAmbientCubeColour[ normIsNegative.y + 2 ].xyz + \
													normSquared.z * cAmbientCubeColour[ normIsNegative.z + 4 ].xyz; \
		LPFLOAT ShadowLightLevel = cAmbientCubeColour[ 0 ].w;

	#if defined(_LIGHTING_DYNAMIC_)
		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION \
			_output.colour = _input.colour; \
			/*_output.colour = float4( 1.0f, 1.0f, 1.0f, 1.0f );*/ \
			LPFLOAT3 normSquared = normal * normal; \
			int3 normIsNegative = ( normal < 0.0f ); \
			_output.colour.xyz *=	normSquared.x * cAmbientCubeColour[ normIsNegative.x ].xyz + \
														normSquared.y * cAmbientCubeColour[ normIsNegative.y + 2 ].xyz + \
														normSquared.z * cAmbientCubeColour[ normIsNegative.z + 4 ].xyz; \
			_output.eye.w = cAmbientCubeColour[ 0 ].w;

	#elif defined(_LIGHTING_PRECALCULATED_)

		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION	DO_VS_AMBIENT_LIGHTING_CALCULATIONS

	#else

		#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION	DO_VS_AMBIENT_LIGHTING_CALCULATIONS

	#endif

	#define DO_PS_AMBIENT_LIGHTING_CALCULATIONS( __accumulator__ )


#else // AMBIENT TYPE

	//
	// Fallback "no ambient lighting" version
	//
	#define DECLARE_AMBIENT_LIGHTING_PARAMS
	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_VS( __index__ )
	#define DECLARE_AMBIENT_LIGHTING_INTERPOLATORS_PS( __index__ )
	#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS
	#define DO_VS_AMBIENT_LIGHTING_CALCULATIONS_TRANSITION
	#define DO_PS_AMBIENT_LIGHTING_CALCULATIONS( __accumulator__ )

#endif // AMBIENT TYPE

#endif	// defined( _3DSMAX_ )


#endif	//  _AMBIENT_LIGHTING_GLOBALS_FXH
